unit SFPRawView;

{
  A viewer for XML documents.
  Based on SAX2 interfaces, using SAX for Pascal.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 5 November, 2002.
}

interface

{$I SAX.inc}

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, ComCtrls, Grids, TypInfo,
{$IFDEF DELPHI4}
  ImgList,
{$ENDIF}
{$IFDEF DELPHI6_UP}
  Variants,
{$ENDIF}
  SAX, SAXExt, SAXMS, SAXKW, SAXCS, SAXDE;

type
  TfrmSAX2Viewer = class(TForm, IContentHandler, IDeclHandler, IDTDHandler,
      IEntityResolver, IErrorHandler, ILexicalHandler)
    pgcMain: TPageControl;
      tshStructure: TTabSheet;
        trvXML: TTreeView;
        pgcDetails: TPageControl;
          tshDocument: TTabSheet;
            Label1: TLabel;
            edtDocType: TEdit;
            Label2: TLabel;
            edtPublicId: TEdit;
            Label3: TLabel;
            edtSystemId: TEdit;
            Label6: TLabel;
            stgEntities: TStringGrid;
            Label7: TLabel;
            stgNotations: TStringGrid;
          tshElement: TTabSheet;
            pnlNames: TPanel;
              Label4: TLabel;
              edtURI: TEdit;
              Label5: TLabel;
              edtLocalName: TEdit;
            stgAttributes: TStringGrid;
            stgPrefixes: TStringGrid;
          tshText: TTabSheet;
            lblNodeType: TLabel;
            memText: TMemo;
      tshSource: TTabSheet;
        memSource: TRichEdit;
    mnuMain: TMainMenu;
      mniFile: TMenuItem;
        mniOpen: TMenuItem;
        mniSep1: TMenuItem;
        mniVendor: TMenuItem;
        mniParserOptions: TMenuItem;
          mniValidation: TMenuItem;
          mniNamespaces: TMenuItem;
          mniNamespacePrefixes: TMenuItem;
        mniSep2: TMenuItem;
        mniExit: TMenuItem;
      mniView: TMenuItem;
        mniExpandAll: TMenuItem;
        mniCollapseAll: TMenuItem;
        mniSep3: TMenuItem;
        mniViewSource: TMenuItem;
    imlXML: TImageList;
    dlgOpen: TOpenDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mniOpenClick(Sender: TObject);
    procedure mniVendorClick(Sender: TObject);
    procedure mniParserOptionClick(Sender: TObject);
    procedure mniExitClick(Sender: TObject);
    procedure mniExpandAllClick(Sender: TObject);
    procedure mniCollapseAllClick(Sender: TObject);
    procedure mniViewSourceClick(Sender: TObject);
    procedure trvXMLChange(Sender: TObject; Node: TTreeNode);
  private
    FCharIcon: Integer;
    FContentHandler: IContentHandler;
    FCurrent: TTreeNode;
    FDeclHandler: IDeclHandler;
    FDTDHandler: IDTDHandler;
    FEntityResolver: IEntityResolver;
    FErrorHandler: IErrorHandler;
    FLexicalHandler: ILexicalHandler;
    FLocator: ILocator;
    FPrefixes: TStringList;
    FXMLReader: IXMLReader;
    procedure ClearTree;
    procedure LoadDoc(Filename: string);
    procedure ShowError(const Level: TMsgDlgType; const Error: ISAXParseError);
    function TruncateText(Text: string): string;
  public
    // IContentHandler
    procedure characters(const ch: SAXString);
    procedure endDocument();
    procedure endElement(const uri, localName, qName: SAXString);
    procedure endPrefixMapping(const prefix: SAXString);
    procedure ignorableWhitespace(const ch: SAXString);
    procedure processingInstruction(const target, data: SAXString);
    procedure setDocumentLocator(const locator: ILocator);
    procedure skippedEntity(const name: SAXString);
    procedure startDocument();
    procedure startElement(
      const uri, localName, qName: SAXString; const atts: IAttributes);
    procedure StartPrefixMapping(const prefix, uri: SAXString);
    // IDeclHandler
    procedure attributeDecl(const eName, aName, attrType, mode, value: SAXString);
    procedure elementDecl(const name, model: SAXString);
    procedure externalEntityDecl(const name, publicId, systemId: SAXString);
    procedure internalEntityDecl(const name, value: SAXString);
    // IDTDHandler
    procedure notationDecl(const name, publicId, systemId: SAXString);
    procedure unparsedEntityDecl(
      const name, publicId, systemId, notationName: SAXString);
    // IEntityResolver
    function resolveEntity(const publicId, systemId: SAXString): IInputSource;
    // IErrorHandler
    procedure error(const e: ISAXParseError);
    procedure fatalError(const e: ISAXParseError);
    procedure warning(const e: ISAXParseError);
    // ILexicalHandler
    procedure comment(const ch: SAXString);
    procedure endCData();
    procedure endDTD();
    procedure endEntity(const name: SAXString);
    procedure startCData();
    procedure startDTD(const name, publicId, systemId: SAXString);
    procedure startEntity(const name: SAXString);
  end;

var
  frmSAX2Viewer: TfrmSAX2Viewer;

implementation

{$R *.DFM}

resourcestring
  AttributeDeclDesc = 'Attribute Declaration';
  AttributeDecln    = '%s %s %s';
  AttributeDesc     = 'Attribute';
  CDataDesc         = 'CDATA Section';
  CommentDesc       = 'Comment';
  DTDDeclDesc       = 'DTD Declaration';
  DTDDecln          = '%s PUBLIC "%s" SYSTEM "%s"';
  DTDDesc           = 'DTD';
  ElementDeclDesc   = 'Element Declaration';
  EntityDecln       = 'PUBLIC "%s" SYSTEM "%s"';
  EntityDesc        = 'Entity Declaration/Reference';
  ErrorLocation     = '(%d) %s'#13#10'at line %d, column %d';
  InstructionDesc   = 'Processing Instruction';
  NameDesc          = 'Name';
  NotationDesc      = 'Notation';
  PrefixDesc        = 'Prefix';
  PublicDesc        = 'Public Id';
  SkippedDesc       = 'Skipped entity';
  SystemDesc        = 'System Id';
  TextDesc          = 'Text';
  URIDesc           = 'URI';
  ValueDesc         = 'Value';

const
  { Icons for tree view }
  DocumentIcon      = 0;
  ElementIcon       = 1;
  ProcInstrIcon     = 2;
  TextIcon          = 3;
  CDataIcon         = 4;
  CommentIcon       = 5;
  DTDIcon           = 6;
  ElementDeclIcon   = 7;
  AttributeDeclIcon = 8;
  EntityIcon        = 9;

{ TString ---------------------------------------------------------------------}

type
  { Wrapper around a string }
  TString = class(TObject)
  private
    FValue: string;
  public
    constructor Create(Value: string);
    property Value: string read FValue write FValue;
  end;

{ Initialisation }
constructor TString.Create(Value: string);
begin
  inherited Create;
  FValue := Value;
end;

{ TElement --------------------------------------------------------------------}

type
  { Details about an element }
  TElement = class(TObject)
  private
    FAttributes: TStringList;
    FLocalName: string;
    FPrefixes: TStringList;
    FURI: string;
  public
    constructor Create(URI, LocalName: string);
    destructor Destroy; override;
    property Attributes: TStringList read FAttributes write FAttributes;
    property LocalName: string read FLocalName write FLocalName;
    property Prefixes: TStringList read FPrefixes write FPrefixes;
    property URI: string read FURI write FURI;
  end;

{ Initialisation }
constructor TElement.Create(URI, LocalName: string);
begin
  inherited Create;
  FURI        := URI;
  FLocalName  := LocalName;
  FAttributes := TStringList.Create;
  FPrefixes   := TStringList.Create;
end;

{ Release resources }
destructor TElement.Destroy;
begin
  FAttributes.Free;
  FPrefixes.Free;
  inherited Destroy;
end;

{ TfrmSAX2Viewer --------------------------------------------------------------}

{ Initialisation - load list of vendors on start up }
procedure TfrmSAX2Viewer.FormCreate(Sender: TObject);
var
  Vendors: TStringList;
  Index, DefIndex: Integer;
  MenuItem: TMenuItem;
begin
  FContentHandler    := Self;
  FDeclHandler       := Self;
  FDTDHandler        := Self;
  FEntityResolver    := Self;
  FErrorHandler      := Self;
  FLexicalHandler    := Self;
  FPrefixes          := TStringList.Create;
  FXMLReader         := nil;
  dlgOpen.InitialDir := ExtractFilePath(Application.ExeName);
  with stgEntities do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
    Cells[3, 0] := NotationDesc;
  end;
  with stgNotations do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
  end;
  with stgAttributes do
  begin
    Cells[0, 0] := AttributeDesc;
    Cells[1, 0] := ValueDesc;
  end;
  with stgPrefixes do
  begin
    Cells[0, 0] := PrefixDesc;
    Cells[1, 0] := URIDesc;
  end;
  Vendors := TStringList.Create;
  try
    ListSAXVendors(Vendors);
    DefIndex := 0;
    for Index := 0 to Vendors.Count - 1 do
    begin
      MenuItem := TMenuItem.Create(Self);
      with MenuItem do
      begin
        Caption    := Vendors[Index];
        GroupIndex := 1;
        RadioItem  := True;
        OnClick    := mniVendorClick;
        if Caption = DefaultSAXVendor then
          DefIndex := Index;
      end;
      mniVendor.Add(MenuItem);
    end;
    mniVendorClick(mniVendor.Items[DefIndex]);
  finally
    Vendors.Free;
  end;
  pgcDetails.ActivePage := tshDocument;
end;

{ Release resources }
procedure TfrmSAX2Viewer.FormDestroy(Sender: TObject);
begin
  ClearTree;
  FLocator := nil;
  FPrefixes.Free;
  FXMLReader := nil;
end;

{ Empty the tree of associated objects }
procedure TfrmSAX2Viewer.ClearTree;
var
  Index: Integer;
begin
  for Index := 0 to trvXML.Items.Count - 1 do
    if Assigned(trvXML.Items[Index].Data) then
      TObject(trvXML.Items[Index].Data).Free;
  trvXML.OnChange := nil;
  trvXML.Items.Clear;
  trvXML.OnChange := trvXMLChange;
end;

{ Load an XML document }
procedure TfrmSAX2Viewer.LoadDoc(Filename: string);
begin
  Screen.Cursor := crHourGlass;
  try
    trvXML.Items.BeginUpdate;
    try
      pgcDetails.ActivePage := tshDocument;
      { Load the source document }
      memSource.Lines.LoadFromFile(Filename);
      dlgOpen.Filename      := Filename;
      { Attempt to set standard features from menu items }
      try
        FXMLReader.Features[ValidationFeature] := mniValidation.Checked;
      except on E: ESAXException do
        mniValidation.Enabled := False;
      end;
      try
        FXMLReader.Features[NamespacesFeature] := mniNamespaces.Checked;
      except on E: ESAXException do
        mniNamespaces.Enabled := False;
      end;
      try
        FXMLReader.Features[NamespacePrefixesFeature] := mniNamespacePrefixes.Checked;
      except on E: ESAXException do
        mniNamespacePrefixes.Enabled := False;
      end;
      { Parse the document - form already registered with parser as handlers }
      FXMLReader.parse(Filename);
    finally
      trvXML.Items.EndUpdate;
    end;
  finally
    Screen.Cursor := crDefault;
  end;
end;

{ Select a file to open }
procedure TfrmSAX2Viewer.mniOpenClick(Sender: TObject);
begin
  with dlgOpen do
    if Execute then
      LoadDoc(Filename);
end;

{ Select a vendor to use }
procedure TfrmSAX2Viewer.mniVendorClick(Sender: TObject);

  function RemoveAccelerator(const Value: string): string;
  begin
    Result := Value;
    Delete(Result, Pos('&', Result), 1);
  end;

var
  Vendor: TSAXVendor;
  DeclProp, LexicalProp: IInterfaceProperty;
begin
  (Sender as TMenuItem).Checked := True;
  Vendor     := GetSAXVendor(RemoveAccelerator((Sender as TMenuItem).Caption));
  Caption    := 'Raw SAX for Pascal (' + Vendor.Description + ')';
  FXMLReader := Vendor.XMLReader;
  FXMLReader.ContentHandler := FContentHandler;
  FXMLReader.DTDHandler     := FDTDHandler;
  FXMLReader.EntityResolver := FEntityResolver;
  FXMLReader.ErrorHandler   := FErrorHandler;
  try
    DeclProp := FXMLReader.Properties[DeclHandlerProperty] as IInterfaceProperty;
    DeclProp.Value := FDeclHandler;
  except on E: ESAXException do
    // Ignore
  end;
  try
    LexicalProp := FXMLReader.Properties[LexicalHandlerProperty] as IInterfaceProperty;
    LexicalProp.Value := FLexicalHandler;
  except on E: ESAXException do
    // Ignore
  end;
  mniValidation.Enabled        := True;
  mniNamespaces.Enabled        := true;
  mniNamespacePrefixes.Enabled := True;
  pgcDetails.ActivePage := tshDocument;
  ClearTree;
end;

{ Toggle the parser options }
procedure TfrmSAX2Viewer.mniParserOptionClick(Sender: TObject);
begin
  with TMenuItem(Sender) do
    Checked := not Checked;
end;

{ Exit the application }
procedure TfrmSAX2Viewer.mniExitClick(Sender: TObject);
begin
  Close;
end;

{ Expand all nodes below the current one }
procedure TfrmSAX2Viewer.mniExpandAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Expand(True);
end;

{ Collapse all nodes below the current one }
procedure TfrmSAX2Viewer.mniCollapseAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Collapse(True);
end;

{ Toggle between structure and source }
procedure TfrmSAX2Viewer.mniViewSourceClick(Sender: TObject);
begin
  mniViewSource.Checked := not mniViewSource.Checked;
  if mniViewSource.Checked then
    pgcMain.ActivePage := tshSource
  else
    pgcMain.ActivePage := tshStructure;
end;

{ Show an error message and position cursor in source text }
procedure TfrmSAX2Viewer.ShowError(const Level: TMsgDlgType;
  const Error: ISAXParseError);
var
  Line, SelStart: Integer;
  XMLSource: string;
begin
  MessageDlg(Format(ErrorLocation, [0, Error.getMessage,
    Error.getLineNumber, Error.getColumnNumber]), Level, [mbOK], 0);
  mniViewSource.Checked := True;
  pgcMain.ActivePage    := tshSource;
  SelStart              := 0;
  XMLSource             := memSource.Lines.Text;
  for Line := 1 to Error.getLineNumber - 1 do
  begin
    SelStart := SelStart + Pos(#10, XMLSource);
    Delete(XMLSource, 1, Pos(#10, XMLSource));
  end;
  memSource.SelStart  := SelStart + Error.getColumnNumber - 1;
  memSource.SelLength := 0;
end;

{ Truncate text to a given length }
function TfrmSAX2Viewer.TruncateText(Text: string): string;
begin
  if Length(Text) > 20 then
    Result := Copy(Text, 1, 17) + '...'
  else
    Result := Text;
end;

{ Display details for the selected XML element }
procedure TfrmSAX2Viewer.trvXMLChange(Sender: TObject; Node: TTreeNode);
var
  Index: Integer;
  Element: TElement;
begin
  if not Assigned(trvXML.Selected.Data) then
    { Document }
    pgcDetails.ActivePage := tshDocument
  else if TObject(trvXML.Selected.Data) is TString then
  begin
    { Text/processing instruction/etc }
    pgcDetails.ActivePage := tshText;
    case trvXML.Selected.ImageIndex of
      AttributeDeclIcon: lblNodeType.Caption := AttributeDeclDesc;
      CDataIcon:         lblNodeType.Caption := CDataDesc;
      CommentIcon:       lblNodeType.Caption := CommentDesc;
      DTDIcon:           lblNodeType.Caption := DTDDeclDesc;
      ElementDeclIcon:   lblNodeType.Caption := ElementDeclDesc;
      EntityIcon:        lblNodeType.Caption := EntityDesc;
      ProcInstrIcon:     lblNodeType.Caption := InstructionDesc;
      else               lblNodeType.Caption := TextDesc;
    end;
    memText.Lines.Text := TString(trvXML.Selected.Data).Value;
  end
  else if TObject(trvXML.Selected.Data) is TElement then
  begin
    { Element }
    pgcDetails.ActivePage := tshElement;
    Element               := TElement(trvXML.Selected.Data);
    edtURI.Text           := Element.URI;
    edtLocalName.Text     := Element.LocalName;
    with stgAttributes do
    begin
      if Element.Attributes.Count = 0 then
        RowCount := 2
      else
        RowCount := Element.Attributes.Count + 1;
      Rows[1].Clear;
      for Index := 0 to Element.Attributes.Count - 1 do
      begin
        Cells[0, Index + 1] := Element.Attributes.Names[Index];
        Cells[1, Index + 1] :=
          Element.Attributes.Values[Element.Attributes.Names[Index]];
      end;
    end;
    with stgPrefixes do
    begin
      if Element.Prefixes.Count = 0 then
        RowCount := 2
      else
        RowCount := Element.Prefixes.Count + 1;
      Rows[1].Clear;
      for Index := 0 to Element.Prefixes.Count - 1 do
      begin
        Cells[0, Index + 1] := Element.Prefixes.Names[Index];
        Cells[1, Index + 1] :=
          Element.Prefixes.Values[Element.Prefixes.Names[Index]];
      end;
    end;
  end;
end;

{ SAX handler implementations -------------------------------------------------}

{ IContentHandler -------------------------------------------------------------}

{ Add a text node to the tree }
procedure TfrmSAX2Viewer.characters(const ch: SAXString);
var
  Index: Integer;
  Text: string;
begin
  { Ignore all white space }
  Text := ch;
  for Index := 1 to Length(Text) do
    if Text[Index] > ' ' then
      Break;
  if Index > Length(Text) then
    Exit;

  with trvXML.Items.AddChildObject(FCurrent, TruncateText(ch),
    TString.Create(ch)) do
  begin
    ImageIndex    := FCharIcon;
    SelectedIndex := FCharIcon;
  end;
end;

{ Tidy up and expand the top level of the tree }
procedure TfrmSAX2Viewer.endDocument();
begin
  trvXML.Items[0].Expand(False);
end;

{ Move the current context up the hierarchy when an element ends }
procedure TfrmSAX2Viewer.endElement(const uri, localName, qName: SAXString);
begin
  FCurrent := FCurrent.Parent;
end;

{ Note end of prefix mapping }
procedure TfrmSAX2Viewer.endPrefixMapping(const prefix: SAXString);
begin
  { Do nothing }
end;

{ As the name says - ignore this }
procedure TfrmSAX2Viewer.ignorableWhitespace(const ch: SAXString);
begin
  { Do nothing }
end;

{ Add a processing instruction to the tree }
procedure TfrmSAX2Viewer.processingInstruction(const target, data: SAXString);
begin
  with trvXML.Items.AddChildObject(FCurrent, target, TString.Create(data)) do
  begin
    ImageIndex    := ProcInstrIcon;
    SelectedIndex := ProcInstrIcon;
  end;
end;

{ Save the locator for later }
procedure TfrmSAX2Viewer.setDocumentLocator(const locator: ILocator);
begin
  FLocator := locator;
end;

{ Add a skipped entity to the tree }
procedure TfrmSAX2Viewer.skippedEntity(const name: SAXString);
begin
  with trvXML.Items.AddChildObject(FCurrent, name, TString.Create('')) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ Initialisation for a new document display }
procedure TfrmSAX2Viewer.StartDocument();
begin
  ClearTree;
  FCharIcon              := TextIcon;
  FCurrent               := trvXML.Items.AddChild(nil, dlgOpen.FileName);
  FCurrent.ImageIndex    := DocumentIcon;
  FCurrent.SelectedIndex := DocumentIcon;
  edtDocType.Text        := '';
  edtPublicId.Text       := '';
  edtSystemId.Text       := dlgOpen.FileName;
  stgEntities.RowCount   := 2;
  stgEntities.Rows[1].Clear;
  stgNotations.RowCount  := 2;
  stgNotations.Rows[1].Clear;
end;

{ Note this element as the current node and save its attributes }
procedure TfrmSAX2Viewer.startElement(
  const uri, localName, qName: SAXString; const atts: IAttributes);
var
  Element: TElement;
  Index: Integer;
begin
  Element := TElement.Create(uri, localName);
  for Index := 0 to atts.Length - 1 do
    Element.Attributes.Values[atts.getQName(Index)] :=
      atts.getValue(Index);
  Element.Prefixes.Assign(FPrefixes);
  FPrefixes.Clear;
  FCurrent               :=
    trvXML.Items.AddChildObject(FCurrent, qName, Element);
  FCurrent.ImageIndex    := ElementIcon;
  FCurrent.SelectedIndex := ElementIcon;
  if edtDocType.Text = '' then
    edtDocType.Text := qName;
end;

{ Save prefix for display with element }
procedure TfrmSAX2Viewer.startPrefixMapping(const prefix, uri: SAXString);
begin
  FPrefixes.Values[prefix] := uri;
end;

{ IDeclHandler ----------------------------------------------------------------}

{ Add an attribute declaration to the tree }
procedure TfrmSAX2Viewer.attributeDecl(
  const eName, aName, attrType, mode, value: SAXString);
begin
  with trvXML.Items.AddChildObject(FCurrent, eName + '.' + aName,
    TString.Create(Format(AttributeDecln, [attrType, mode, value]))) do
  begin
    ImageIndex    := AttributeDeclIcon;
    SelectedIndex := AttributeDeclIcon;
  end;
end;

{ Add an element declaration to the tree }
procedure TfrmSAX2Viewer.elementDecl(const name, model: SAXString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, name, TString.Create(model)) do
  begin
    ImageIndex    := ElementDeclIcon;
    SelectedIndex := ElementDeclIcon;
  end;
end;

{ Add an entity declaration to the tree }
procedure TfrmSAX2Viewer.externalEntityDecl(
  const name, publicId, systemId: SAXString);
begin
  with trvXML.Items.AddChildObject(FCurrent, name,
    TString.Create(Format(EntityDecln, [publicId, systemId]))) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ Add an entity declaration to the tree }
procedure TfrmSAX2Viewer.internalEntityDecl(const name, value: SAXString);
begin
  with trvXML.Items.AddChildObject(FCurrent, name, TString.Create(value)) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ IDTDHandler -----------------------------------------------------------------}

{ Save the details about a notation }
procedure TfrmSAX2Viewer.notationDecl(const name, publicId, systemId: SAXString);
var
  NotationName: SAXString;
begin
  NotationName := name;
  with stgNotations do
  begin
    if Cells[0, 1] <> '' then
      RowCount := RowCount + 1;
    Cells[0, RowCount - 1] := NotationName;
    Cells[1, RowCount - 1] := publicId;
    Cells[2, RowCount - 1] := systemId;
  end;
end;

{ Save the details about an external entity }
procedure TfrmSAX2Viewer.unparsedEntityDecl(
  const name, publicId, systemId, notationName: SAXString);
var
  EntityName: SAXString;
begin
  EntityName := name;
  with stgEntities do
  begin
    if Cells[0, 1] <> '' then
      RowCount := RowCount + 1;
    Cells[0, RowCount - 1] := EntityName;
    Cells[1, RowCount - 1] := publicId;
    Cells[2, RowCount - 1] := systemId;
    Cells[3, RowCount - 1] := notationName;
  end;
end;

{ IEntityResolver -------------------------------------------------------------}

{ Find the source for an entity }
function TfrmSAX2Viewer.resolveEntity(const publicId, systemId: SAXString):
  IInputSource;
begin
  Result := nil;
end;

{ IErrorHandler ---------------------------------------------------------------}

procedure TfrmSAX2Viewer.error(const e: ISAXParseError);
begin
  ShowError(mtError, e);
end;

procedure TfrmSAX2Viewer.fatalError(const e: ISAXParseError);
begin
  ShowError(mtError, e);
end;

procedure TfrmSAX2Viewer.warning(const e: ISAXParseError);
begin
  ShowError(mtWarning, e);
end;

{ ILexicalHandler -------------------------------------------------------------}

{ Add a comment to the tree }
procedure TfrmSAX2Viewer.comment(const ch: SAXString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, TruncateText(ch), TString.Create(ch)) do
  begin
    ImageIndex    := CommentIcon;
    SelectedIndex := CommentIcon;
  end;
end;

{ Note end of CDATA section }
procedure TfrmSAX2Viewer.endCData();
begin
  FCharIcon := TextIcon;
end;

{ Move the current context up the hierarchy when the DTD ends }
procedure TfrmSAX2Viewer.endDTD();
begin
  FCurrent := FCurrent.Parent;
end;

{ Move the current context up the hierarchy when an entity reference ends }
procedure TfrmSAX2Viewer.endEntity(const name: SAXString);
begin
  FCurrent := FCurrent.Parent;
end;

{ Note start of CDATA section - text returned through characters method }
procedure TfrmSAX2Viewer.startCData();
begin
  FCharIcon := CDataIcon;
end;

{ Add a DTD declaration to the tree }
procedure TfrmSAX2Viewer.startDTD(const name, publicId, systemId: SAXString);
begin
  FCurrent := trvXML.Items.AddChildObject(FCurrent, DTDDesc,
    TString.Create(Format(DTDDecln, [name, publicId, systemId])));
  with FCurrent do
  begin
    ImageIndex    := DTDIcon;
    SelectedIndex := DTDIcon;
  end;
end;

{ Add an entity reference to the tree }
procedure TfrmSAX2Viewer.startEntity(const name: SAXString);
begin
  FCurrent := trvXML.Items.AddChildObject(FCurrent, name, TString.Create(name));
  with FCurrent do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

end.
